/*  Errechnet die Dauer der Überlappungen des gegebenen AGs auf der Hauptressource mit anderen AGen.

   Je nach Wert des Parameters _filter_for_other_ab2 werden nur bestimmete Überlappungen mit dem gegebenen AG beachtet:
   'dlz_a2_et' - Überlappungen vom gegebenen DLZ-terminierten AG mit anderen AGen, welche früher beendet sind oder bei gleichzeitigem Ende mit dem gegeben AG, welche eine kleinere a2_id haben
   'dlz_ti_id' - Überlappungen vom gegebenen DLZ-terminierten AG mit anderen AGen, welche früher in der Tabelle scheduling.resource_timeline eingetragen wurden (ti_ids sind kleiner als die ti_ids des gegeben AG)
   'none'      - Überlappungen vom gegebenen AG mit allen anderen AGen
*/
SELECT tsystem.function__drop_by_regex( 'resource_timeline__overlap_duration__by__a2_id', 'scheduling', _commit => true );
CREATE OR REPLACE FUNCTION scheduling.resource_timeline__overlap_duration__by__a2_id(
    _a2_id                integer,                  -- Der gegebene AG
    _filter_for_other_ab2 varchar DEFAULT 'none'    -- Filter der anderen AGe:
                                                    --    + 'none'      - alle anderen Überlappenden AGe werden beachtet
                                                    --    + 'dlz_a2_et' - von den anderen DLZ-terminierten AGen werden nur AGe beachtet, welche eine kleinere a2_et haben oder bvei gleicher a2_et, welche eine kleinere a2_id haben
                                                    --    + 'dlz_ti_id' - von den anderen DLZ-terminierten AGen werden nur AGe mit kleinerer ti_id beachtet
) RETURNS integer AS $$


    -- Dauer der Überlappungen auf der Hauptressource mit anderen AGen ermitteln.
    -- https://dba.stackexchange.com/questions/100965/combining-separate-ranges-into-largest-possible-contiguous-ranges
    WITH
      _parameters AS (
          SELECT a2_id
              , a2_at
              , a2_et
              , a2w_resource_id_main_terminated AS resource_id_HR
              , min( ti_id ) AS ti_id_min
            FROM ab2
            JOIN ab2_wkstplan ON a2w_a2_id = a2_id
            JOIN scheduling.resource_timeline ON ti_a2_id = a2_id
                                            AND ti_resource_id = a2w_resource_id_main_terminated
                                            AND ti_type IN ( 'task', 'task.buffer' )
                                            AND (
                                                      ti_stat IS NOT DISTINCT FROM 'dlz'  -- Gegebener AG muss DLZ-terminiert sein, außer ...
                                                  OR _filter_for_other_ab2 = 'none'       -- ... Parameter _filter_for_other_ab2 ist 'none'.
                                                )
          WHERE a2_id = _a2_id
          GROUP BY a2_id, a2_at, a2_et, a2w_resource_id_main_terminated
      ),
      _other_ti_ranges AS (  -- Zeitbereiche aller Task-Einträge anderer AGe, welche mit dem übergebenen AG überlappen und deren Abarbeitung des AG gleichzeitig oder eher fertig wird.
          SELECT ti_id
              , tsrange( ti_date_start, ti_date_end ) AS ti_range
              , ti_type
              , ti_ta_kf
              , ti_usage
            FROM scheduling.resource_timeline
            JOIN ab2 ON a2_id = ti_a2_id
          CROSS JOIN _parameters
          WHERE ti_type IN ( 'task', 'task.blocktime', 'task.buffer' )           -- Nur Task- und Blocktime-Einträge bei den überlappenden AGen interessant. Die Blocktimes der überlappenden AGe sind interessant, da diese durch die Überlappung nicht mehr die gewünschte Zeit auf der Ressource reservieren.
            AND ti_resource_id =  _parameters.resource_id_HR      -- Nur Einträge auf der gleichen Hauptressource, wie bei dem gegeben AG selektieren.
            AND ti_a2_id       <> _parameters.a2_id               -- Nur Einträge von anderen AGen, als dem gegeben AG auswählen.
            AND ti_date_start  <  _parameters.a2_et               -- Überlappung mit Task-Einträgen des gegebenen AG.
            AND ti_date_end    >  _parameters.a2_at               -- Überlappung mit Task-Einträgen des gegebenen AG.
            AND (
                    ( -- Beachtung der Timeline-Einträge aller anderen AGe
                        (
                            _filter_for_other_ab2 IS null
                        )
                        OR
                        (
                            _filter_for_other_ab2 NOT IN ( 'dlz_a2_et', 'dlz_ti_id' )
                        )
                    )
                    OR
                    ( -- Beachtung der Timeline-Einträge nach a2_et bei DLZ-terminierten AGen
                        _filter_for_other_ab2 = 'dlz_a2_et'
                        AND
                        (
                            (
                                ab2.a2_et   <  _parameters.a2_et    -- Abarbeitung des AG wird eher fertig, als beim gegebenen AG.
                            )
                            OR
                            (
                                ( ab2.a2_et =  _parameters.a2_et )  -- Abarbeitung des AG wird gleichzeitig mit gegebenenAG fertig ...
                                AND                                 -- ... und ...
                                ( ab2.a2_id < _parameters.a2_id )   -- ... AG-ID ist kleiner als beim gegebenen AG.
                            )
                            OR
                            (
                                ti_stat IS DISTINCT FROM 'dlz'      -- Alle NICHT DLZ-terminierten Timeline-Einträge.
                            )
                        )
                    )
                    OR
                    ( -- Beachtung der Timeline-Einträge nach ti_id bei DLZ-terminierten AGen
                        _filter_for_other_ab2 = 'dlz_ti_id'
                        AND
                        (
                            (
                                ti_id   <  _parameters.ti_id_min    -- AG wurde früher als gegebener AG eingetragen.
                            )
                            OR
                            (
                                ti_stat IS DISTINCT FROM 'dlz'      -- Alle NICHT DLZ-terminierten Timeline-Einträge.
                            )
                        )
                    )
                )
            -- AND b.ti_usage + a.ti_usage > 1.0 -- TODO AXS: Nur solche zum übergebenen AG parallelen Timeline-Einträge anderer AG auswählen, welche zu einer überlastung der "ti_usage" führen.
      ),
      _running_max_end AS ( -- Sortierte Bereiche, laufendes Maximum des Enddatums
          SELECT ti_id
              , ti_range
              , lower( ti_range ) AS startdate
              , max( upper( ti_range ) ) OVER ( ORDER BY ti_range ) AS enddate
              , ti_usage
            FROM _other_ti_ranges
      ),
      _gaps AS ( -- Gleiche Sortierung, wie vorher. Prüfung auf Lücken.
          SELECT *
              , lag( enddate ) OVER ( ORDER BY ti_range ) < startdate OR NULL AS gap  -- Wenn das vorhergehende Enddatum früher als das Startdatum ist, dann haben wir eine Lücke.
            FROM _running_max_end
      ),
      _groups AS ( -- Gleiche Sortierung, wie vorher. Zuordnung zu Gruppen.
          SELECT *
              , count( gap ) OVER ( ORDER BY ti_range ) AS grp  -- Gruppen bilden durch "laufendes" Zählen der Lücken.
            FROM _gaps
      ),
      _other_ab2 AS ( -- Bereiche der Gruppen bilden.
          SELECT tsrange( min( startdate ), max( enddate ) ) AS ti_range
              , min( ti_usage ) AS ti_usage
            FROM _groups
          GROUP BY grp
          ORDER BY 1
      ),
      _given_ab2 AS ( -- Bereiche des übergebenen AG.
          SELECT ti_id
              , tsrange( ti_date_start, ti_date_end ) AS ti_range
              , ti_ta_kf
            FROM scheduling.resource_timeline
          CROSS JOIN _parameters
          WHERE ti_resource_id = _parameters.resource_id_HR
            AND ti_a2_id = _parameters.a2_id
            AND ti_type IN ( 'task', 'task.buffer' ) --  Beim gegebenen AG interessieren uns keine Blocktimes, nur die Task-Einträge. Ansonsten wird ggf. mehr Zeit bei den Überlappungen reserviert, als überhaupt verplant ist.
      )
    SELECT  coalesce(
                extract( epoch FROM
                    sum(
                        (
                            upper( g.ti_range * o.ti_range )  -- Ender der Überlappung
                          - lower( g.ti_range * o.ti_range )  -- Begin der Überlappung
                        ) * g.ti_ta_kf                        -- zugehöriger Korrekturfaktor des entsprechenden Timeline-Eintrags des übergebenen AG
                    )
                )
              , 0
            ) AS _overlap_duration
     FROM _given_ab2 AS g                                               -- Timeline-Einträge des übergebenen AG
     JOIN _other_ab2 AS o ON lower( o.ti_range ) < upper( g.ti_range )  -- Timeline-Einträge anderer AG, welchemit dem übergebenen AG überlappen
                         AND upper( o.ti_range ) > lower( g.ti_range )

$$ LANGUAGE sql VOLATILE;